<?php
// gerar_banner.php
// Script para geração de banners usando a API do TMDB

// --- Configurações ---
$api_key = '2557fea3b40cbbd5a65e12422d017de1'; // Sua API Key do TMDB
$api_base_url = 'https://api.themoviedb.org/3/';
$image_base_url = 'https://image.tmdb.org/t/p/original';
$output_dir = __DIR__ . '/../img/'; // Diretório de saída dos banners

// Verificar se a biblioteca GD está disponível
if (!function_exists('imagecreate')) {
    die("Biblioteca GD não instalada. Por favor, instale-a para continuar.");
}

// --- Função para tratamento de erros ---
function handleError($message) {
    error_log($message);
    die($message);
}

// --- Função para chamar a API do TMDB com tratamento de erros ---
function callTmdbApi($endpoint, $api_key) {
    $url = "https://api.themoviedb.org/3/{$endpoint}?api_key={$api_key}&language=pt-BR";
    
    $ch = curl_init($url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    
    $response = curl_exec($ch);
    
    if (curl_errno($ch)) {
        handleError('Erro cURL: ' . curl_error($ch));
    }
    
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    if ($httpCode !== 200) {
        handleError("Erro na requisição. Código HTTP: {$httpCode}");
    }
    
    return json_decode($response, true);
}

// --- Processamento dos parâmetros GET ---
$id = filter_input(INPUT_GET, 'id', FILTER_VALIDATE_INT);
$type = filter_input(INPUT_GET, 'tipo', FILTER_SANITIZE_STRING);
$layout = filter_input(INPUT_GET, 'layout', FILTER_SANITIZE_STRING) ?? 'vertical';

if (!$id || !$type) {
    handleError("Parâmetros inválidos.");
}

// Validar tipo de mídia
if (!in_array($type, ['filme', 'serie'])) {
    handleError("Tipo de mídia inválido.");
}

// Endpoint dinâmico baseado no tipo
$endpoint = ($type === 'filme') ? "movie/{$id}" : "tv/{$id}";
$data = callTmdbApi($endpoint, $api_key);

// --- Validações adicionais ---
if (!$data || isset($data['success']) && $data['success'] === false) {
    handleError("Dados não encontrados ou ID inválido.");
}

// --- Coleta e tratamento de dados ---
$title = mb_convert_case($data['title'] ?? $data['name'] ?? 'Título Desconhecido', MB_CASE_UPPER, 'UTF-8');
$poster_path = $data['poster_path'] ?? null;
$overview = $data['overview'] ?? 'Sinopse não disponível.';
$release_date = $data['release_date'] ?? $data['first_air_date'] ?? '';
$year = !empty($release_date) ? substr($release_date, 0, 4) : 'Ano Desconhecido';

// Tratamento de gêneros
$genres = array_map(function($genre) { return $genre['name']; }, $data['genres'] ?? []);
$genres_str = !empty($genres) ? implode(', ', $genres) : 'Gênero Desconhecido';

// --- Configurações de renderização ---
$banner_width = $layout === 'vertical' ? 720 : 1280;
$banner_height = $layout === 'vertical' ? 1080 : 720;

// Criar imagem
$image = imagecreatetruecolor($banner_width, $banner_height);

// Definir cores
$bg_color = imagecolorallocate($image, 16, 16, 24);
$text_color = imagecolorallocate($image, 255, 255, 255);
$shadow_color = imagecolorallocate($image, 0, 0, 0);
$background_overlay_color = imagecolorallocatealpha($image, 0, 0, 0, 60);

// Preencher fundo
imagefill($image, 0, 0, $bg_color);

// Carregar e adicionar poster
if ($poster_path) {
    $poster_url = $image_base_url . $poster_path;
    $poster_data = @file_get_contents($poster_url);
    
    if ($poster_data) {
        $poster_image = imagecreatefromstring($poster_data);
        $poster_width = imagesx($poster_image);
        $poster_height = imagesy($poster_image);

        // Redimensionar poster
        $dest_poster_width = $banner_width * 0.7;
        $dest_poster_height = ($poster_height / $poster_width) * $dest_poster_width;
        
        // Posicionamento
        $dest_x = ($banner_width - $dest_poster_width) / 2;
        $dest_y = $layout === 'vertical' ? 50 : ($banner_height - $dest_poster_height) / 2;

        // Copiar poster
        imagecopyresampled(
            $image, 
            $poster_image, 
            $dest_x, $dest_y, 
            0, 0, 
            $dest_poster_width, $dest_poster_height, 
            $poster_width, $poster_height
        );

        // Liberar memória
        imagedestroy($poster_image);
    }
}

// Adicionar textos
$font_path = __DIR__ . '/../fonts/Inter-Bold.ttf'; // Caminho para fonte personalizada

// Configurações de texto
$title_size = $layout === 'vertical' ? 40 : 30;
$info_size = $layout === 'vertical' ? 25 : 20;
$synopsis_size = $layout === 'vertical' ? 20 : 16;

// Posicionamento do texto
$text_x = $banner_width * 0.1;
$text_y = $banner_height * 0.7;

// Adicionar título
imagettftext(
    $image, 
    $title_size, 
    0, 
    $text_x, 
    $text_y, 
    $text_color, 
    $font_path, 
    $title
);

// Adicionar informações (ano e gênero)
$info_text = "{$year} · {$genres_str}";
$text_y += 50;
imagettftext(
    $image, 
    $info_size, 
    0, 
    $text_x, 
    $text_y, 
    $text_color, 
    $font_path, 
    $info_text
);

// Adicionar sinopse
$text_y += 50;
$wrapped_synopsis = wordwrap($overview, 50, "\n", true);
imagettftext(
    $image, 
    $synopsis_size, 
    0, 
    $text_x, 
    $text_y, 
    $text_color, 
    $font_path, 
    $wrapped_synopsis
);

// Gerar variações de banner
$filename_base = "{$id}_{$type}_Banner";

// Banner padrão
$output_path = $output_dir . $filename_base . '.png';
imagepng($image, $output_path);

// Banner para Status do WhatsApp (quadrado)
$status_image = imagecreatetruecolor(1080, 1080);
imagecopyresampled(
    $status_image, 
    $image, 
    0, 0, 
    0, 0, 
    1080, 1080, 
    $banner_width, $banner_height
);
imagepng($status_image, $output_dir . $filename_base . '_status.png');

// Banner para Instagram (story)
$insta_image = imagecreatetruecolor(1080, 1920);
imagecopyresampled(
    $insta_image, 
    $image, 
    0, 0, 
    0, 0, 
    1080, 1920, 
    $banner_width, $banner_height
);
imagepng($insta_image, $output_dir . $filename_base . '_insta.png');

// Liberar memória
imagedestroy($image);
imagedestroy($status_image);
imagedestroy($insta_image);

// Log de geração de banner
error_log("Banner gerado: {$filename_base}");

// Redirecionar para página de gerenciamento
header("Location: /admin/gerenciar_banners.php?banner_filename=" . urlencode($filename_base . '.png'));
exit;
?>
